// pvi.c: Signetics 2636 PVI emulation module (platform-independent)

// #define DEBUGSPRITES

/* This is how each frame is currently emulated:

now VRST is turned off
  0..  19 are  20 "margin" (above grid but visible) Y-pixels
  20..219 are 200 "grid"                            Y-pixels
 220..271 are  52 "margin" (below grid but visible) Y-pixels
now VRST is turned on
 272..291 are  20 "vertical retrace" (below grid)   Y-pixels
the true frame boundary is here
 292..311 are  20 "vertical retrace" (above grid)   Y-pixels

Almost but not quite the same as shown in the Interton VC 4000 Coding
 Guide. Doing it the same the Coding Guide doesn't seem to help.
*/

#ifdef AMIGA
    #include "lint.h"
    #include "amiga.h"
#endif
#ifdef WIN32
    #include "ibm.h"
#endif

#include "aa.h"
#include "pvi.h"

#include <stdio.h>
#include <string.h>

#define I_XSIZE 227
#define HIDDEN_X  4 /* opposite of XMARGIN
(ie. Arcadia emulator has overscan but Interton/Elektor emulator has
underscan) */

EXPORT UWORD i_spritedata[4] = { I_SPRITE0,
                                 I_SPRITE1,
                                 I_SPRITE2,
                                 I_SPRITE3 };
MODULE struct
{   int   delay,
          state,
          x;
    UBYTE y;// MUST be UBYTE!
} sprite[4];

// to convert PVI -> UVI colours:
MODULE UBYTE from_i[16] =
{   0, //  0 ->  0 white
    1, //  1 ->  1 yellow
    4, //  2 ->  4 purple
    5, //  3 ->  5 red
    2, //  4 ->  2 cyan
    3, //  5 ->  3 green
    6, //  6 ->  6 yellow
    7, //  7 ->  7 black
    8, //  8 ->  8
    9, //  9 ->  9
   10, // 10 -> 10
   11, // 11 -> 11
   12, // 12 -> 12
   13, // 13 -> 13
   14, // 14 -> 14
   15, // 15 -> 15
};

MODULE UBYTE  OutputBuffer[9];
MODULE SLONG  x,
              y,
              xx,
              yy;
MODULE UBYTE  bx,
              by,
              digitcolour,
              result,
              t,
              fgc;
MODULE SLONG  latch_a,
              latch_start;

// from gui.c
IMPORT UBYTE  screen[BOXWIDTH][BOXHEIGHT];
IMPORT ULONG  analog,
              autofire,
              collisions,
              flagline,
              paused,
              swapped,
              warp;
IMPORT FLAG   consoleopen,
              crippled,
              inframe,
              interrupt,
              limitrefresh,
              loop,
              runtorastline,
              runtoframe,
              trace;
IMPORT UBYTE* IOBuffer;
IMPORT FILE*  MacroHandle;
IMPORT TEXT   friendly[2 + 13 + 1 + 1];
IMPORT int    cpl,
              fudge,
              frameskip,
              machine,
              offset,
              overcalc, // must be signed!
              primary,
              rast,
              rastn,
              recmode,
              recsize,
              slice,
              wp;
IMPORT SLONG  ax[2], // analog paddle X-coords
              ay[2]; // analog paddle Y-coords
IMPORT UBYTE  psu,
              memory[32768];
IMPORT UWORD  page,
              iar;
IMPORT ULONG  elapsed,
              frames,
			  downframes,
              totalframes;

#ifdef WIN32
    #ifdef MODE_SHALLOW
        IMPORT UBYTE     display[EVENBOXWIDTH * BOXHEIGHT];
    #else
        IMPORT int       display[BOXWIDTH * BOXHEIGHT];
        IMPORT ULONG     pens[16];
    #endif
    IMPORT UBYTE         sx[2], sy[2];
#endif

#ifdef AMIGA
    IMPORT FLAG          fullscreen;
    IMPORT LONG          pens[16];
    IMPORT int           px, py,
                         wide,
                         size;
    IMPORT UBYTE*        lineptr[MAXBOXHEIGHT];
#endif

MODULE __inline void drawgrid(void);
MODULE void drawdigit(int x, int y, int d, int line);
MODULE void drawsprite(int whichsprite);
MODULE void i_playerinput(UBYTE source, UBYTE dest);
MODULE __inline void i_emuinput(void);
MODULE void pviwrite(UWORD address, UBYTE data);
MODULE __inline void breakrastline(void);
MODULE __inline void i_drawpixel(SLONG x, SLONG y, UBYTE colour);

// Each "character" is 12*20 pixels (3*5 "chunky 4*4" pixels).
MODULE const char led[20][12 + 1] =
{   "aaaabbbbcccc",
    "aaaabbbbcccc",
    "aaaabbbbcccc",
    "aaaabbbbcccc",
    "llll    dddd",
    "llll    dddd",
    "llll    dddd",
    "llll    dddd",
    "kkkkmmmmeeee",
    "kkkkmmmmeeee",
    "kkkkmmmmeeee",
    "kkkkmmmmeeee",
    "jjjj    ffff",
    "jjjj    ffff",
    "jjjj    ffff",
    "jjjj    ffff",
    "iiiihhhhgggg",
    "iiiihhhhgggg",
    "iiiihhhhgggg",
    "iiiihhhhgggg"
};

MODULE UBYTE colltable[I_XSIZE];
/* bits are:
   7:    grid
   6..3: sprites #3..#0 respectively
   2..0: sprite foreground colour (wire OR'ed) */

EXPORT void pvi(void)
{   int whichsprite;

    // assert(machine == ARCADIA);

    inframe = TRUE;

    // assert(!paused);

    // beginning of frame?

    pviwrite(I_BGCOLLIDE ,     0); // clear sprite complete and sprite-bg collision bits
    pviwrite(I_SPRITECOLLIDE , 0); // clear VRST and sprite-sprite collision bits
    pviwrite(I_BGCOLLIDE2,     0); // clear sprite complete and sprite-bg collision bits
    pviwrite(I_SPRITECOLLIDE2, 0); // clear VRST and sprite-sprite collision bits
    pviwrite(I_BGCOLLIDE3,     0); // clear sprite complete and sprite-bg collision bits
    pviwrite(I_SPRITECOLLIDE3, 0); // clear VRST and sprite-sprite collision bits
    pviwrite(I_BGCOLLIDE4,     0); // clear sprite complete and sprite-bg collision bits
    pviwrite(I_SPRITECOLLIDE4, 0); // clear VRST and sprite-sprite collision bits
 
    psu &= ~(PSU_S);
    i_emuinput();

    for (whichsprite = 0; whichsprite <= 3; whichsprite++)
    {   sprite[whichsprite].y        = memory[i_spritedata[whichsprite] + 12]; // SPRITEnY1
        sprite[whichsprite].state    = 0;
    }

    for (rast = 0; rast < 312; rast++)
    {   // assert(!paused); // this assertion fails!

        breakrastline();

        // assert(!paused); // this assertion fails!
        slice = cpl - overcalc;
        cpu();
        CHECKINPUT;

        if (rast < I_BOXHEIGHT)
        {   for (x = 0; x < I_XSIZE; x++)
            {   colltable[x] = 0;
            }

            drawgrid();

            // draw score
            if (memory[I_SCORECTRL] & 1)
            {   y = 200;
            } else
            {   y =  20;
            }

            if (rast == y)
            {   if (memory[I_BGCOLOUR] & 8)
                {   digitcolour = ((memory[I_BGCOLOUR] >> 4) & 7);
                } else
                {   digitcolour = 7; // white
            }   }
            // this is cached for eg. LASERATT game over screen

            if ((rast >= y) && (rast < y + 20)) // 20 rastlines is the height of each digit
            {   drawdigit(32 - HIDDEN_X + 27, y, memory[I_SCORELT] >>   4, rast - y);
                drawdigit(32 - HIDDEN_X + 43, y, memory[I_SCORELT] &  0xf, rast - y);

                x =       32 - HIDDEN_X + 59;
                if (!(memory[I_SCORECTRL] & 2))
                {   x += 16;
                }
                drawdigit(x     , y, memory[I_SCORERT] >>   4, rast - y);
                drawdigit(x + 16, y, memory[I_SCORERT] &  0xf, rast - y);
        }   }

        // sprites raster must be drawn AFTER the background raster

        fgc = ((memory[I_SPR01COLOURS] >> 3) & 7);
        drawsprite(0);
        fgc = ( memory[I_SPR01COLOURS]       & 7);
        drawsprite(1);
        fgc = ((memory[I_SPR23COLOURS] >> 3) & 7);
        drawsprite(2);
        fgc = ( memory[I_SPR23COLOURS]       & 7);
        drawsprite(3);

        // Once a collision is registered, does it remain set
        // for the remainder of the frame, or only the current
        // rastline? (We're assuming it's for the remainder of
        // the frame.)

        if (collisions)
        {   t = memory[I_SPRITECOLLIDE];
            for (x = 0; x < I_XSIZE; x++)                   // bitwise...                    3210,g,fgc
            {   if ((colltable[x] & 0x30) == 0x30) t |= 32; // sprites #0 and #1 collision (%0011,0,000)
                if ((colltable[x] & 0x50) == 0x50) t |= 16; // sprites #0 and #2 collision (%0101,0,000)
                if ((colltable[x] & 0x90) == 0x90) t |=  8; // sprites #0 and #3 collision (%1001,0,000)
                if ((colltable[x] & 0x60) == 0x60) t |=  4; // sprites #1 and #2 collision (%0110,0,000)
                if ((colltable[x] & 0xA0) == 0xA0) t |=  2; // sprites #1 and #3 collision (%1010,0,000)
                if ((colltable[x] & 0xC0) == 0xC0) t |=  1; // sprites #2 and #3 collision (%1100,0,000)
            }
            pviwrite(I_SPRITECOLLIDE , t);

            t = memory[I_SPRITECOLLIDE2];
            for (x = 0; x < I_XSIZE; x++)                   // bitwise...                    3210,g,fgc
            {   if ((colltable[x] & 0x30) == 0x30) t |= 32; // sprites #0 and #1 collision (%0011,0,000)
                if ((colltable[x] & 0x50) == 0x50) t |= 16; // sprites #0 and #2 collision (%0101,0,000)
                if ((colltable[x] & 0x90) == 0x90) t |=  8; // sprites #0 and #3 collision (%1001,0,000)
                if ((colltable[x] & 0x60) == 0x60) t |=  4; // sprites #1 and #2 collision (%0110,0,000)
                if ((colltable[x] & 0xA0) == 0xA0) t |=  2; // sprites #1 and #3 collision (%1010,0,000)
                if ((colltable[x] & 0xC0) == 0xC0) t |=  1; // sprites #2 and #3 collision (%1100,0,000)
            }
            pviwrite(I_SPRITECOLLIDE2, t);

            t = memory[I_SPRITECOLLIDE3];
            for (x = 0; x < I_XSIZE; x++)                   // bitwise...                    3210,g,fgc
            {   if ((colltable[x] & 0x30) == 0x30) t |= 32; // sprites #0 and #1 collision (%0011,0,000)
                if ((colltable[x] & 0x50) == 0x50) t |= 16; // sprites #0 and #2 collision (%0101,0,000)
                if ((colltable[x] & 0x90) == 0x90) t |=  8; // sprites #0 and #3 collision (%1001,0,000)
                if ((colltable[x] & 0x60) == 0x60) t |=  4; // sprites #1 and #2 collision (%0110,0,000)
                if ((colltable[x] & 0xA0) == 0xA0) t |=  2; // sprites #1 and #3 collision (%1010,0,000)
                if ((colltable[x] & 0xC0) == 0xC0) t |=  1; // sprites #2 and #3 collision (%1100,0,000)
            }
            pviwrite(I_SPRITECOLLIDE3, t);

            t = memory[I_SPRITECOLLIDE4];
            for (x = 0; x < I_XSIZE; x++)                   // bitwise...                    3210,g,fgc
            {   if ((colltable[x] & 0x30) == 0x30) t |= 32; // sprites #0 and #1 collision (%0011,0,000)
                if ((colltable[x] & 0x50) == 0x50) t |= 16; // sprites #0 and #2 collision (%0101,0,000)
                if ((colltable[x] & 0x90) == 0x90) t |=  8; // sprites #0 and #3 collision (%1001,0,000)
                if ((colltable[x] & 0x60) == 0x60) t |=  4; // sprites #1 and #2 collision (%0110,0,000)
                if ((colltable[x] & 0xA0) == 0xA0) t |=  2; // sprites #1 and #3 collision (%1010,0,000)
                if ((colltable[x] & 0xC0) == 0xC0) t |=  1; // sprites #2 and #3 collision (%1100,0,000)
            }
            pviwrite(I_SPRITECOLLIDE4, t);

            if (rast >= 20 && rast <= 219) // just a speed optimization
            {   t = memory[I_BGCOLLIDE];
                for (x = 0; x < I_XSIZE; x++)
                {   if ((colltable[x] & 0x18) == 0x18) t |= 128; // sprite #0 collision with background (%0001,1,000)
                    if ((colltable[x] & 0x28) == 0x28) t |=  64; // sprite #1 collision with background (%0010,1,000)
                    if ((colltable[x] & 0x48) == 0x48) t |=  32; // sprite #2 collision with background (%0100,1,000)
                    if ((colltable[x] & 0x88) == 0x88) t |=  16; // sprite #3 collision with background (%1000,1,000)
                }
                pviwrite(I_BGCOLLIDE , t);

                t = memory[I_BGCOLLIDE2];
                for (x = 0; x < I_XSIZE; x++)
                {   if ((colltable[x] & 0x18) == 0x18) t |= 128; // sprite #0 collision with background (%0001,1,000)
                    if ((colltable[x] & 0x28) == 0x28) t |=  64; // sprite #1 collision with background (%0010,1,000)
                    if ((colltable[x] & 0x48) == 0x48) t |=  32; // sprite #2 collision with background (%0100,1,000)
                    if ((colltable[x] & 0x88) == 0x88) t |=  16; // sprite #3 collision with background (%1000,1,000)
                }
                pviwrite(I_BGCOLLIDE2, t);

                t = memory[I_BGCOLLIDE3];
                for (x = 0; x < I_XSIZE; x++)
                {   if ((colltable[x] & 0x18) == 0x18) t |= 128; // sprite #0 collision with background (%0001,1,000)
                    if ((colltable[x] & 0x28) == 0x28) t |=  64; // sprite #1 collision with background (%0010,1,000)
                    if ((colltable[x] & 0x48) == 0x48) t |=  32; // sprite #2 collision with background (%0100,1,000)
                    if ((colltable[x] & 0x88) == 0x88) t |=  16; // sprite #3 collision with background (%1000,1,000)
                }
                pviwrite(I_BGCOLLIDE3, t);

                t = memory[I_BGCOLLIDE4];
                for (x = 0; x < I_XSIZE; x++)
                {   if ((colltable[x] & 0x18) == 0x18) t |= 128; // sprite #0 collision with background (%0001,1,000)
                    if ((colltable[x] & 0x28) == 0x28) t |=  64; // sprite #1 collision with background (%0010,1,000)
                    if ((colltable[x] & 0x48) == 0x48) t |=  32; // sprite #2 collision with background (%0100,1,000)
                    if ((colltable[x] & 0x88) == 0x88) t |=  16; // sprite #3 collision with background (%1000,1,000)
                }
                pviwrite(I_BGCOLLIDE4, t);
        }   }

        if (rast == 271)
        {   // this point is the start of vertical blank

            pviwrite(I_SPRITECOLLIDE , (UBYTE) (memory[I_SPRITECOLLIDE ] | 0x40)); // set VRST bit
            pviwrite(I_SPRITECOLLIDE2, (UBYTE) (memory[I_SPRITECOLLIDE2] | 0x40)); // set VRST bit
            pviwrite(I_SPRITECOLLIDE3, (UBYTE) (memory[I_SPRITECOLLIDE3] | 0x40)); // set VRST bit
            pviwrite(I_SPRITECOLLIDE4, (UBYTE) (memory[I_SPRITECOLLIDE4] | 0x40)); // set VRST bit

            psu |= PSU_S;

            if (!(psu & PSU_II))
            {   interrupt = TRUE;
    }   }   }

    // We are assuming that the actual diagonal (bottom-right to top-left)
    // beam retrace is done during rasters 309..311.

    // Currently, frame skipping only affects warp mode when "limit
    // refresh rate" is off.

    if
    (   (!warp || !limitrefresh)
     && (frames % frameskip == 0)
    )
    {   updatescreen();
    }

    frames++;

    inframe = FALSE;
    if (crippled)
    {   uncripple();
    }

    if (recmode == RECMODE_PLAY && offset >= recsize)
    {   if (loop)
        {   macro_restartplayback();
        } else
        {   macro_stop();
    }   }

    if (runtoframe)
    {   runtoframe = FALSE;
        OPENCONSOLE;
        printf("Reached next frame (%ld).\n\n", frames);
        REACTIVATE;
        pause();
}   }

MODULE __inline void breakrastline(void)
{   if (runtorastline)
    {   runtorastline = FALSE;
        OPENCONSOLE;
        printf("Reached next raster line (%ld).\n\n", rast);
        REACTIVATE;
        runto();
    }
    if (rast == rastn)
    {   rastn = -1;
        OPENCONSOLE;
        printf("Reached raster line %ld.\n\n", rast);
        REACTIVATE;
        runto();
}   }

MODULE __inline void drawgrid(void)
{   PERSIST int gridline[220] =                                                         // Interton/Elektor screen is vertically as follows:
    {   20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, //   0.. 19 top (visible) margin
         0,  0,                                                                         //  20.. 21 thin
                 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, //  22.. 39 thick
         2,  2,                                                                         //  40.. 41 thin
                 3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3, //  42.. 59 thick
         4,  4,                                                                         //  60.. 61 thin
                 5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5, //  62.. 79 thick
         6,  6,                                                                         //  80.. 81 thin
                 7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7, //  82.. 99 thick
         8,  8,                                                                         // 100..101 thin
                 9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9, // 102..119 thick
        10, 10,                                                                         // 120..121 thin
                11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, // 122..139 thick
        12, 12,                                                                         // 140..141 thin
                13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, // 142..159 thick
        14, 14,                                                                         // 160..161 thin
                15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, // 162..179 thick
        16, 16,                                                                         // 180..181 thin
                17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, // 182..199 thick
        18, 18,                                                                         // 200..201 thin
                19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19  // 202..219 thick
                                                                                        // 220..271 bottom visible margin
                                                                                        // 272..311 bottom invisible margin
    };
    AUTO int i, j, k,
             m,
             w;

    // assert(rast < BOXHEIGHT);

#ifdef DEBUGSPRITES
//    OPENCONSOLE;
//    printf("Drawing grid for rastline %ld.\n", rast);
//    REACTIVATE;
#endif

    // blank the current rastline
    // colltable[] has already been cleared
    if (flagline)
    {   if (memory[I_BGCOLOUR] & 8) // background/grid enable
        {   for (x = 0; x < BOXWIDTH; x++)
            {   i_drawpixel
                (   x,
                    rast,
                    (UBYTE) (8 + (memory[I_BGCOLOUR] & 7))
                );
        }   }
        else
        {   for (x = 0; x < BOXWIDTH; x++)
            {   i_drawpixel
                (   x,
                    rast,
                    8 // dark black (black)
                );
    }   }   }
    else
    {   if (memory[I_BGCOLOUR] & 8) // background/grid enable
        {   for (x = 0; x < BOXWIDTH; x++)
            {   i_drawpixel
                (   x,
                    rast,
                    (UBYTE) (7 - (memory[I_BGCOLOUR] & 7))
                );
        }   }
        else
        {   for (x = 0; x < BOXWIDTH; x++)
            {   i_drawpixel
                (   x,
                    rast,
                    7 // black
                );
    }   }   }

    if (rast < 20 || rast > 219 || !(memory[I_BGCOLOUR] & 8))
    {   return;
    }

    i = gridline[rast];                 //  i       is     0..19
    k = memory[I_HORIZGRID + (i >> 2)]; // (i >> 2) yields 0.. 4
    switch (k & 0xC0)
    {
    case  0x40:
        w = 2;
    acase 0xC0:
        w = 4;
        adefault: // $00, $80
        w = 1;
    break;
    }

    switch (i & 3)                      // (i &  3) yields 0.. 3
    {
    case 0:
        if (k & 1)
        {   w = 8;
        }
    acase 1:
        if (((rast - 20) % 40) <= 10)
        {   if (k & 2)
            {   w = 8;
        }   }
        else
        {   if (k & 4)
            {   w = 8;
        }   }
    acase 2:
        if (k & 8)
        {   w = 8;
        }
    acase 3:
        if (((rast - 20) % 40) <= 30)
        {   if (k & 16)
            {   w = 8;
        }   }
        else
        {   if (k & 32)
            {   w = 8;
        }   }
    break;
    }

    // j goes   0,  1,  2,  3,  4,  5,  6,  7,   8,   9,  10,  11,  12,  13,  14,  15
    // m goes 128, 64, 32, 16,  8,  4,  2,  1, 128,  64,  32,  16,   8,   4,   2,   1
    // x goes  31, 41, 49, 57, 65, 73, 81, 89,  97, 105, 113, 121, 129, 137, 145, 153
    if (flagline)
    {   for
        (   x = 31, j = 0, m = 128;
            j < 16;
            j++, x += 8, m >>= 1
        )
        {   if (memory[I_VERTGRID + (i * 2) + (j >> 3)] & m)
            {   for (xx = x; xx < x + w; xx++)
                {   colltable[xx] |= 8;
                    i_drawpixel
                    (   xx - HIDDEN_X,
                        rast,
                        (UBYTE) (8 + ((memory[I_BGCOLOUR] >> 4) & 7))
                    );
            }   }
            if (j == 7)
            {   m = 256;
    }   }   }
    else
    {   for
        (   x = 31, j = 0, m = 128;
            j < 16;
            j++, x += 8, m >>= 1
        )
        {   if (memory[I_VERTGRID + (i * 2) + (j >> 3)] & m)
            {   for (xx = x; xx < x + w; xx++)
                {   colltable[xx] |= 8;
                    i_drawpixel
                    (   xx - HIDDEN_X,
                        rast,
                        (UBYTE) (7 - ((memory[I_BGCOLOUR] >> 4) & 7))
                    );
            }   }
            if (j == 7)
            {   m = 256;
}   }   }   }

/* Gridlines that are the same colour as the background (ie.
   invisible) don't register collisions.

     -0-  -1-  -2-  -3-  -4-  -5-  -6-  -7-  -8-  -9-  -A-  -B-  -C-  -D-  -E-  -F-
abc  ###  ..#  ###  ###  #.#  ###  ###  ###  ###  ###  ###  #..  ###  ..#  ###  ###
l d  # #  . #  ..#  . #  # #  # .  # .  . #  # #  # #  # #  # .  # .  . #  # .  # .
kme  #.#  ..#  ###  ###  ###  ###  ###  ..#  ###  ###  ###  ###  #..  ###  ###  ###
j f  # #  . #  #..  . #  . #  . #  # #  . #  # #  . #  # #  # #  # .  # #  # .  # .
ihg  ###  ..#  ###  ###  ..#  ###  ###  ..#  ###  ###  #.#  ###  ###  ###  ###  #..

Apparently the Interton/Elektor handles $A..$F values by drawing nothing.
The capital $B looks the same as 8
and capital $D looks the same as 0
so we use lowercase "b" and "d". */

MODULE void drawdigit(int x, int y, int d, int line)
{   // bit $1000: 'm' segment
    // bit $0800: 'l' segment
    // bit $0400: 'k' segment
    // bit $0200: 'j' segment
    // bit $0100: 'i' segment
    // bit $0080: 'h' segment
    // bit $0040: 'g' segment
    // bit $0020: 'f' segment
    // bit $0010: 'e' segment
    // bit $0008: 'd' segment
    // bit $0004: 'c' segment
    // bit $0002: 'b' segment
    // bit $0001: 'a' segment
    PERSIST const UWORD digit_to_segment[10] = // [16] for hex support
    {   0x0fff, // 0
        0x007c, // 1
        0x17df, // 2
        0x15ff, // 3
        0x1c7d, // 4
        0x1df7, // 5
        0x1ff7, // 6
        0x007f, // 7
        0x1fff, // 8
        0x1dff  // 9
/*    , 0x1f7f, // A
        0x1ff1, // B
        0x0fc7, // C
        0x17fc, // D
        0x1fd7, // E
        0x1f17  // F
*/
    };
    int j;

        // assert(d >= 0 && d <= 15);
    if (d >= 10)
        {       return;
        }

    for (j = 0; j < 12 + 1; j++)
    {   if
        (   digit_to_segment[d] & (1 << (led[line][j] - 'a'))
        )
        {   colltable[x + j - HIDDEN_X] |= 8;
            i_drawpixel(x + j, y + line, digitcolour);
}   }   }

/* This draws one raster of one sprite (ie. sprites are drawn on a per-
rastline basis, not at the end of the frame). */
MODULE void drawsprite(int whichsprite)
{   AUTO    int   i;
    AUTO    UBYTE data;
    PERSIST int   thesize[4],
                  x2; // faster if PERSISTent
 // PERSIST UBYTE spriteimage[4][10];

    // assert(rast < 272);

    /* delay is used to stretch out the sprite horizontally (it counts up
       until it reaches thesize) */

#ifdef DEBUGSPRITES
    OPENCONSOLE;
    printf("Rastline %ld, sprite %ld, state %ld.\n", rast, whichsprite, sprite[whichsprite].state);
    REACTIVATE;
#endif

    switch (sprite[whichsprite].state)
    {
    case 0: // top of sprite not yet reached by the beam
        if (rast == sprite[whichsprite].y + 1) // was +0 in V4.24 and +2 in V4.25
        {   sprite[whichsprite].delay = 0;
            sprite[whichsprite].state = 1; // starts drawing immediately (this rastline)
        } else
        {   break;
        }
    case 1: // drawing the top row
    case 2: // drawing the 2nd row
    case 3: // etc.
    case 4:
    case 5:
    case 6:
    case 7:
    case 8:
    case 9:
    case 10:
/* Rereading I_SIZES every rastline for the original sprites (as we do
   here) helps for INVADERA/INVADERB? */
        if (whichsprite == 0)
        {   thesize[0] = 1 << ( memory[I_SIZES]       & 3);
        } elif (whichsprite == 1)
        {   thesize[1] = 1 << ((memory[I_SIZES] >> 2) & 3);
        } elif (whichsprite == 2)
        {   thesize[2] = 1 << ((memory[I_SIZES] >> 4) & 3);
        } else
        {   // assert(whichsprite == 3);
            thesize[3] = 1 << ((memory[I_SIZES] >> 6) & 3);
        }
        xx = memory[i_spritedata[whichsprite] + 10]; // SPRITEnX1
        if (xx <= 190) // maximum x-pixel for starting to draw a sprite
        {   data = memory[i_spritedata[whichsprite]
                 + sprite[whichsprite].state
                 - 1];
            // data = spriteimage[whichsprite][sprite[whichsprite].state - 1];
#ifdef DEBUGSPRITES
            OPENCONSOLE;
            printf
            (   "Drawing row %ld of sprite %ld (1st) at %ld,%ld (image data is $%lX).\n",
                sprite[whichsprite].state,
                whichsprite,
                xx, // it is given in real Interton/Elektor coords, not i_drawpixel()-adjusted
                rast,
                data
            );
            REACTIVATE;
#endif
            for (i = 0; i < 8; i++) // the 8 columns of the sprite
            {   for (x = 0; x < thesize[whichsprite]; x++)
                {   x2 = xx + (i * thesize[whichsprite]) + x;
                    // assert(x2 >= 0);
                    if (x2 < I_XSIZE)
                    {   if (data & (128 >> i))
                        {   colltable[x2] |= (16 << whichsprite) | fgc;
                            if (rast < BOXHEIGHT && x2 >= HIDDEN_X && x2 < HIDDEN_X + BOXWIDTH)
                            {   i_drawpixel
                                (   x2 - HIDDEN_X,
                                    rast,
                                    (UBYTE) (colltable[x2] & 7)
                                );
        }   }   }   }   }   }

        sprite[whichsprite].delay++;
        if (sprite[whichsprite].delay >= thesize[whichsprite])
        {   sprite[whichsprite].delay = 0;
            sprite[whichsprite].state++;
        }
        if (sprite[whichsprite].state == 11)
        {   pviwrite(I_BGCOLLIDE , (UBYTE) (memory[I_BGCOLLIDE ] | (8 >> whichsprite)));
            pviwrite(I_BGCOLLIDE2, (UBYTE) (memory[I_BGCOLLIDE2] | (8 >> whichsprite)));
            pviwrite(I_BGCOLLIDE3, (UBYTE) (memory[I_BGCOLLIDE3] | (8 >> whichsprite)));
            pviwrite(I_BGCOLLIDE4, (UBYTE) (memory[I_BGCOLLIDE4] | (8 >> whichsprite)));
            if (!(psu & PSU_II))
            {   interrupt = TRUE;
            }
            // strictly speaking, the interrupt should happen on the next
            // CPU instruction, not at the end of the rastline?
        }
    break;
    case 11:
        // old sprite has been drawn, now we reread SPRITEnY2
        sprite[whichsprite].y = memory[i_spritedata[whichsprite] + 13]; // SPRITEnY2
        sprite[whichsprite].y++; // overflow OK!

        if (sprite[whichsprite].y <= 252) // $00..$FC
        {   sprite[whichsprite].delay = 0;
            sprite[whichsprite].state = 12;
        } else // $FD..$FF
        {   break;
        }
    case 12:
        // these figures are probably wrong by now.
		// 220 is the highest that draughts   will tolerate
        // 224 is the lowest  that casino     will tolerate
        // 251 is the highest that casino     (interton) will tolerate
        // 228 is the highest that piano      will tolerate
        // 245 is the lowest  that shootgal   will tolerate
        // 248 is the lowest  that invadera/b will tolerate
        // 267 is the highest that pinballa/b will tolerate
        // <= 271 would probably be more accurate (last raster before vblank)?
        // count down the vertical gap
        if (sprite[whichsprite].y == 0 && rast <= fudge)
        {   sprite[whichsprite].state = 13; // starts drawing immediately (this rastline)
        } else
        {   sprite[whichsprite].y--;
            break;
        }
    case 13:
    case 14:
    case 15:
    case 16:
    case 17:
    case 18:
    case 19:
    case 20:
    case 21:
    case 22:
/* Rereading I_SIZES every rastline for the duplicate sprites (as we do
   here) helps for BACKGAMM, and is bad for CIRCUS. */
        if (whichsprite == 0)
        {   thesize[0] = 1 << ( memory[I_SIZES]       & 3);
        } elif (whichsprite == 1)
        {   thesize[1] = 1 << ((memory[I_SIZES] >> 2) & 3);
        } elif (whichsprite == 2)
        {   thesize[2] = 1 << ((memory[I_SIZES] >> 4) & 3);
        } else
        {   // assert(whichsprite == 3);
            thesize[3] = 1 << ((memory[I_SIZES] >> 6) & 3);
        }
        xx = memory[i_spritedata[whichsprite] + 11]; // SPRITEnX2
        if (xx <= 190) // maximum x-pixel for starting to draw a sprite
        {   data = memory[i_spritedata[whichsprite]
                 + sprite[whichsprite].state
                 - 13];
            // data = spriteimage[whichsprite][sprite[whichsprite].state - 13];
#ifdef DEBUGSPRITES
            OPENCONSOLE;
            printf
            (   "Drawing row %ld of sprite %ld (2nd) at %ld,%ld (image data is $%lX).\n",
                sprite[whichsprite].state - 12,
                whichsprite,
                xx, // it is given in real Interton/Elektor coords, not i_drawpixel()-adjusted
                rast,
                data
            );
            REACTIVATE;
#endif
            for (i = 0; i < 8; i++) // the 8 columns of the sprite
            {   for (x = 0; x < thesize[whichsprite]; x++)
                {   x2 = xx + (i * thesize[whichsprite]) + x;
                    // assert(x2 >= 0);
                    if (x2 < I_XSIZE)
                    {   if (data & (128 >> i))
                        {   colltable[x2] |= (16 << whichsprite) | fgc;
                            if (rast < BOXHEIGHT && x2 >= HIDDEN_X && x2 < HIDDEN_X + BOXWIDTH)
                            {   i_drawpixel
                                (   x2 - HIDDEN_X,
                                    rast,
                                    (UBYTE) (colltable[x2] & 7)
                                );
        }   }   }   }   }   }

        sprite[whichsprite].delay++;
        if (sprite[whichsprite].delay >= thesize[whichsprite])
        {   sprite[whichsprite].delay = 0;
            sprite[whichsprite].state++;
        }
        if (sprite[whichsprite].state == 23)
        {   pviwrite(I_BGCOLLIDE , (UBYTE) (memory[I_BGCOLLIDE ] | (8 >> whichsprite)));
            pviwrite(I_BGCOLLIDE2, (UBYTE) (memory[I_BGCOLLIDE2] | (8 >> whichsprite)));
            pviwrite(I_BGCOLLIDE3, (UBYTE) (memory[I_BGCOLLIDE3] | (8 >> whichsprite)));
            pviwrite(I_BGCOLLIDE4, (UBYTE) (memory[I_BGCOLLIDE4] | (8 >> whichsprite)));
            if (!(psu & PSU_II))
            {   interrupt = TRUE;
            }
            // strictly speaking, the interrupt should happen on the next CPU instruction,
            // not at the end of the rastline.
            sprite[whichsprite].state = 11;
        }
    adefault:
    break;
}   }

MODULE void pviwrite(UWORD address, UBYTE data)
{   /* Writes to memory from the PVI always go through this
    routine (so that watchpoints work properly). */

    if (address == wp && data != memory[wp])
    {   getfriendly(wp);
        OPENCONSOLE;
        printf("PVI is writing $%X to %s at raster %ld!\n\n", data, friendly, rast);
        REACTIVATE;
        runto();
    }

    memory[address] = data;
}

// This works for both Interton and Elektor
MODULE void i_playerinput(UBYTE source, UBYTE dest)
{   // dest is which side you want to set the registers of.
    // source is which side you want to use to do it.

    if (recmode == RECMODE_PLAY)
    {   if (dest == 0)
        {
#ifdef AMIGA
            pviwrite(I_P1PADDLE    , IOBuffer[offset++]);
#endif
#ifdef WIN32
            pviwrite(I_P1PADDLE    , IOBuffer[offset]);
            if (psu & PSU_F) // paddle interpolation bit
            {   sy[dest] = IOBuffer[offset++];
            } else
            {   sx[dest] = IOBuffer[offset++];
            }
#endif
            pviwrite(I_P1LEFTKEYS  , IOBuffer[offset++]);
            pviwrite(I_P1MIDDLEKEYS, IOBuffer[offset++]);
            pviwrite(I_P1RIGHTKEYS , IOBuffer[offset++]);
        } else
        {   // assert(dest == 1);

#ifdef AMIGA
            pviwrite(I_P2PADDLE    , IOBuffer[offset++]);
#endif
#ifdef WIN32
            pviwrite(I_P2PADDLE    , IOBuffer[offset]);
            if (psu & PSU_F) // paddle interpolation bit
            {   sy[dest] = IOBuffer[offset++];
            } else
            {   sx[dest] = IOBuffer[offset++];
            }
#endif
            pviwrite(I_P2LEFTKEYS  , IOBuffer[offset++]);
            pviwrite(I_P2MIDDLEKEYS, IOBuffer[offset++]);
            pviwrite(I_P2RIGHTKEYS , IOBuffer[offset++]);
    }   }
    else
    {   // assert(recmode == RECMODE_NORMAL || recmode == RECMODE_RECORD);

        xx   =
        yy   = 0;

        result = ReadJoystick((UWORD) (1 - source));

// if primary is 1, secondary is 2
// if primary is 2, secondary is 1
// if primary is 8, secondary is 1

        // left column

        // "1" key (Interton) or "RCAS" key (Elektor)
        t = 0;
        if (primary == 1)
        {   if
            (   (autofire && ((frames % totalframes) < downframes)
             || (result & JOYFIRE1))
             || KeyDown(i_keypads[source][0])
            )
            {   t = 0x80;
        }   }
        else
        {   // assert(primary == 2 || primary == 8);
            if (result & JOYFIRE2)
            {   t = 0x80;
        }   }
        if
        (   KeyDown(i_keypads[source][1])
#ifdef AMIGA
         || (   source == 1
             && KeyDown(SCAN_ND)
            )
#endif
        )
        {   t = 0x80;
        }
        t += (0x40 * KeyDown(i_keypads[source][ 4])); // "4"  key (Interton) or "BP1/2" key (Elektor)
        t += (0x20 * KeyDown(i_keypads[source][ 7])); // "7"  key (Interton) or "PC"    key (Elektor)
        t += (0x10 * KeyDown(i_keypads[source][10])); // "Cl" key (Interton) or "-"     key (Elektor)
        pviwrite((UWORD) (I_P1LEFTKEYS   + (dest * 4)), t);

        // middle column

        // "2" key (Interton) or "WCAS" key (Elektor)
        t = 0;
        if (primary == 2)
        {   if
            (   (autofire && ((frames % totalframes) < downframes)
             || (result & JOYFIRE1))
             || KeyDown(i_keypads[source][0])
            )
            {   t = 0x80;
        }   }
        elif (primary == 1)
        {   if (result & JOYFIRE2)
            {   t = 0x80;
        }   }
        if
        (   KeyDown(i_keypads[source][2])
        )
        {   t = 0x80;
        }
        t += (0x40 * KeyDown(i_keypads[source][ 5])); // "5"  key (Interton) or "REG" key (Elektor)

        if (primary == 8)
        {   if
            (   (autofire && ((frames % totalframes) < downframes)
             || (result & JOYFIRE1))
             || KeyDown(i_keypads[source][0])
            )
            {   t |= 0x20;
        }   }
        t |= (0x20 * KeyDown(i_keypads[source][ 8])); // "8"  key (Interton) or "MEM" key (Elektor)
        // "0" key (Interton) or "+" key (Elektor)
        if
        (   KeyDown(i_keypads[source][11])
         || (result & JOYFIRE4)
        )
        {   t |= 0x10;
        }
        pviwrite((UWORD) (I_P1MIDDLEKEYS + (dest * 4)), t);

        // right column

        // "3" key (Interton) or "C" key (Elektor)
        if
        (   KeyDown(i_keypads[source][ 3])
         || (result & JOYFIRE3)
        )
        {   t = 0x80;
        } else
        {   t = 0;
        }
        t += (0x40 * KeyDown(i_keypads[source][ 6])); // "6"  key (Interton) or "8" key (Elektor)
        t += (0x20 * KeyDown(i_keypads[source][ 9])); // "9"  key (Interton) or "4" key (Elektor)
        t += (0x10 * KeyDown(i_keypads[source][12])); // "En" key (Interton) or "0" key (Elektor)
        pviwrite((UWORD) (I_P1RIGHTKEYS  + (dest * 4)), t);

        // paddle

        if   (result & JOYUP)                 yy = -1;
        elif (result & JOYDOWN)               yy =  1;
        if   (result & JOYLEFT)               xx = -1;
        elif (result & JOYRIGHT)              xx =  1;

        if   (KeyDown(i_keypads[source][13])) yy = -1;
        elif (KeyDown(i_keypads[source][14])) yy =  1;
        if   (KeyDown(i_keypads[source][15])) xx = -1;
        elif (KeyDown(i_keypads[source][16])) xx =  1;

        if (analog)
        {   if   (xx  == -1) { ax[dest] -= 4; if (ax[dest] <   0) ax[dest] =   0; }
            elif (xx  ==  1) { ax[dest] += 4; if (ax[dest] > 255) ax[dest] = 255; }
            if   (yy  == -1) { ay[dest] -= 4; if (ay[dest] <   0) ay[dest] =   0; }
            elif (yy  ==  1) { ay[dest] += 4; if (ay[dest] > 255) ay[dest] = 255; }
            bx = (UBYTE) ax[dest];
            by = (UBYTE) ay[dest];
        } else
        {   if (xx == -1)
            {   bx = 1; // very low is needed for INVADERS
            } elif (xx == 1)
            {   bx = 225;
            } else
			{   if (machine == INTERTON)
                {   bx = 0x70; // >= $70 is needed for SUPERINV
				} else
                {   // assert(machine == ELEKTOR);
                    bx = 0x6F; // <= $6F is needed for JOYSTICK 
            }   }

            if (yy == -1)
            {   by = 1;
            } elif (yy == 1)
            {   by = 225;
            } else
			{   if (machine == INTERTON)
                {   by = 0x70; // >= $70 is needed for SUPERINV
				} else
                {   // assert(machine == ELEKTOR);
                    by = 0x6F; // <= $6F is needed for JOYSTICK 
        }   }   }

        if (psu & PSU_F) // paddle interpolation bit
        {   pviwrite((UWORD) (I_P1PADDLE + dest), by);
#ifdef WIN32
            sy[dest] = by;
#endif
        } else
        {   pviwrite((UWORD) (I_P1PADDLE + dest), bx);
#ifdef WIN32
            sx[dest] = bx;
#endif
}   }   }

// This works for both Interton and Elektor
MODULE __inline void i_emuinput(void)
{   readkybd();

    // must always do i_playerinput(foo, 0) then i_playerinput(foo, 1).
    if (swapped)
    {   i_playerinput(1, 0);
        i_playerinput(0, 1);
    } else
    {   i_playerinput(0, 0);
        i_playerinput(1, 1);
    }
    if (recmode == RECMODE_PLAY)
    {   pviwrite(I_CONSOLE, IOBuffer[offset++]);
    } else
    {   if (machine == INTERTON)
        {   if (KeyDown(SCAN_F1) || latch_start > 0) // START
            {   if (latch_start > 0)
                {   latch_start--;
                }
                t = 0x40;
            } else
            {   t = 0;
            }
            if (KeyDown(SCAN_F2) || latch_a     > 0) // SELECT
            {   latch_a = 0;
                t += 0x80;
        }   }
        else
        {   // assert(machine == ELEKTOR);

            if (KeyDown(SCAN_F1)) t =  0x40; else t = 0; // START
            if (KeyDown(SCAN_F2)) t += 0x80;             // UC
            if (KeyDown(SCAN_F3)) t += 0x20;             // LC
            if (KeyDown(SCAN_F4)) t += 0x10;             // RESET (soft)
        }
        pviwrite(I_CONSOLE, t);
    }

    if (recmode == RECMODE_RECORD)
    {   assert(MacroHandle);

        OutputBuffer[0] = memory[I_P1PADDLE    ];
        OutputBuffer[1] = memory[I_P1LEFTKEYS  ];
        OutputBuffer[2] = memory[I_P1MIDDLEKEYS];
        OutputBuffer[3] = memory[I_P1RIGHTKEYS ];
        OutputBuffer[4] = memory[I_P2PADDLE    ];
        OutputBuffer[5] = memory[I_P2LEFTKEYS  ];
        OutputBuffer[6] = memory[I_P2MIDDLEKEYS];
        OutputBuffer[7] = memory[I_P2RIGHTKEYS ];
        OutputBuffer[8] = memory[I_CONSOLE     ];

        DISCARD fwrite(OutputBuffer, 9, 1, MacroHandle); // should really check return code
        // 9 bytes per frame * 50 frames per second = 450 bytes/sec.
        // Plus, for Elektor, 1 byte every time the E_RANDOMn hardware
        // registers are read.
}   }

#ifdef WIN32

MODULE __inline void i_drawpixel(SLONG x, SLONG y, UBYTE colour)
{   // assert(x >= 0);
    // assert(x < BOXWIDTH);
    // assert(y >= 0);
    // assert(y < BOXHEIGHT);
    // assert(colour >= 0 && colour <= 15);

#ifdef DEBUGSPRITES
    if (x < 0)
    {   OPENCONSOLE;
        printf("X of %ld < 0!\n", x);
        REACTIVATE;
        return;
    }
    if (x >= BOXWIDTH)
    {   OPENCONSOLE;
        printf("X of %ld >= BOXWIDTH of %ld!\n", x, BOXWIDTH);
        REACTIVATE;
        return;
    }
    if (y < 0)
    {   OPENCONSOLE;
        printf("Y of %ld < 0!\n", y);
        REACTIVATE;
        return;
    }
    if (y >= BOXHEIGHT)
    {   OPENCONSOLE;
        printf("Y of %ld >= BOXHEIGHT of %ld!\n", y, BOXHEIGHT);
        REACTIVATE;
        return;
    }
    if (colour > 8)
    {   OPENCONSOLE;
        printf("Illegal colour!\n");
        REACTIVATE;
        return;
    }
#endif

    colour = from_i[colour];

    if (screen[x][y] == colour)
    {   return;
    }

    screen[x][y] = colour;

#ifdef MODE_SHALLOW
    display[x + (y * EVENBOXWIDTH)] = colour;
#else
    display[x + (y * BOXWIDTH)] = pens[colour];
#endif

}

#endif

#ifdef AMIGA

MODULE __inline void i_drawpixel(SLONG x, SLONG y, UBYTE colour)
{   // assert(x >= 0);
    // assert(x < BOXWIDTH);
    // assert(y >= 0);
    // assert(y < BOXHEIGHT);
    // assert(colour >= 0 && colour <= 15);

 /* if (x < 0)
    {   OPENCONSOLE;
        printf("X of %ld < 0!\n", x);
        REACTIVATE;
        return;
    }
    if (x >= BOXWIDTH)
    {   OPENCONSOLE;
        printf("X of %ld >= BOXWIDTH of %ld!\n", x, BOXWIDTH);
        REACTIVATE;
        return;
    }
    if (y < 0)
    {   OPENCONSOLE;
        printf("Y of %ld < 0!\n", y);
        REACTIVATE;
        return;
    }
    if (y >= BOXHEIGHT)
    {   OPENCONSOLE;
        printf("Y of %ld >= BOXHEIGHT of %ld!\n", y, BOXHEIGHT);
        REACTIVATE;
        return;
    }
    if (colour > 8)
    {   OPENCONSOLE;
        printf("Illegal colour!\n");
        REACTIVATE;
        return;
    } */

    colour = from_i[colour];

    if (screen[x][y] == colour)
    {   return;
    }

        screen[x][y] = colour;

    if (fullscreen) // 2*1
    {   // could use pens[colour] but that would be slower

        px = x << 1; // px = x * 2;

        *(lineptr[y] + px++) = colour; // 1,1
        *(lineptr[y] + px)   = colour; // 2,1
    } else
    {   switch(size)
        {
        case 1:
            if (wide == 1) // 1*1
            {   *(lineptr[y     ] + x)    = pens[colour]; // 1,1
            } else
            {   // assert(wide == 2); // 2*1

                px = x << 1; // px = x * 2;

                *(lineptr[y     ] + px++) = pens[colour]; // 1,1
                *(lineptr[y     ] + px)   = pens[colour]; // 2,1
            }
        acase 2:
            py = y << 1; // py = y * 2;

            if (wide == 1) // 2*2
            {   px = x << 1; // px = x * 2;

                *(lineptr[py    ] + px)   = pens[colour]; // 1,1
                *(lineptr[py + 1] + px++) = pens[colour]; // 1,2
                *(lineptr[py    ] + px)   = pens[colour]; // 2,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 2,2
            } else
            {   // assert(wide == 2); // 4*2

                px = x << 2; // px = x * 4;

                *(lineptr[py    ] + px)   = pens[colour]; // 1,1
                *(lineptr[py + 1] + px++) = pens[colour]; // 1,2
                *(lineptr[py    ] + px)   = pens[colour]; // 2,1
                *(lineptr[py + 1] + px++) = pens[colour]; // 2,2
                *(lineptr[py    ] + px)   = pens[colour]; // 3,1
                *(lineptr[py + 1] + px++) = pens[colour]; // 3,2
                *(lineptr[py    ] + px)   = pens[colour]; // 4,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 4,2
            }
        adefault:
            // assert(size == 3);

            px = x * 3 * wide;
            py = y * 3;

            *(lineptr[py    ] + px)   = pens[colour]; // 1,1
            *(lineptr[py + 1] + px)   = pens[colour]; // 1,2
            *(lineptr[py + 2] + px++) = pens[colour]; // 1,3
            *(lineptr[py    ] + px)   = pens[colour]; // 2,1
            *(lineptr[py + 1] + px)   = pens[colour]; // 2,2
            *(lineptr[py + 2] + px++) = pens[colour]; // 2,3
            *(lineptr[py    ] + px)   = pens[colour]; // 3,1
            *(lineptr[py + 1] + px)   = pens[colour]; // 3,2
            *(lineptr[py + 2] + px++) = pens[colour]; // 3,3

            if (wide == 1) return;

            *(lineptr[py    ] + px)   = pens[colour]; // 4,1
            *(lineptr[py + 1] + px)   = pens[colour]; // 4,2
            *(lineptr[py + 2] + px++) = pens[colour]; // 4,3
            *(lineptr[py    ] + px)   = pens[colour]; // 5,1
            *(lineptr[py + 1] + px)   = pens[colour]; // 5,2
            *(lineptr[py + 2] + px++) = pens[colour]; // 5,3
            *(lineptr[py    ] + px)   = pens[colour]; // 6,1
            *(lineptr[py + 1] + px)   = pens[colour]; // 6,2
            *(lineptr[py + 2] + px)   = pens[colour]; // 6,3

        break;
}   }   }

#endif

EXPORT UBYTE __inline loadrand(void)
{   // assert(recmode == RECMODE_PLAY);
    // assert(machine == ELEKTOR);
    return(IOBuffer[offset++]);
}
EXPORT void __inline saverand(UBYTE value)
{   // assert(recmode == RECMODE_RECORD);
    // assert(machine == ELEKTOR);
    OutputBuffer[0] = value;
    DISCARD fwrite(OutputBuffer, 1, 1, MacroHandle); // should really check return code
}

